<?php
require_once __DIR__ . '/../includes/functions.php';

// Verificar autenticação
if (!isAdminAuthenticated()) {
    header('Location: login.php');
    exit;
}

$message = '';
$message_type = '';

// Processar ações
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $nome = sanitize($_POST['nome']);
                $descricao = sanitize($_POST['descricao']);
                
                if (!empty($nome)) {
                    if (createGroup($nome, $descricao)) {
                        $message = 'Grupo criado com sucesso!';
                        $message_type = 'success';
                    } else {
                        $message = 'Erro ao criar grupo.';
                        $message_type = 'danger';
                    }
                } else {
                    $message = 'Nome do grupo é obrigatório.';
                    $message_type = 'warning';
                }
                break;
                
            case 'toggle_status':
                $grupo_id = (int)$_POST['grupo_id'];
                $pdo = getConnection();
                $stmt = $pdo->prepare("UPDATE grupos SET ativo = NOT ativo WHERE id = ?");
                if ($stmt->execute([$grupo_id])) {
                    $message = 'Status do grupo atualizado!';
                    $message_type = 'success';
                } else {
                    $message = 'Erro ao atualizar status.';
                    $message_type = 'danger';
                }
                break;
                
            case 'delete':
                $grupo_id = (int)$_POST['grupo_id'];
                $pdo = getConnection();
                
                // Verificar se há números associados
                $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM numeros WHERE grupo_id = ?");
                $stmt->execute([$grupo_id]);
                $total_numeros = $stmt->fetch()['total'];
                
                if ($total_numeros > 0) {
                    $message = 'Não é possível excluir o grupo pois há números associados a ele.';
                    $message_type = 'warning';
                } else {
                    $stmt = $pdo->prepare("DELETE FROM grupos WHERE id = ?");
                    if ($stmt->execute([$grupo_id])) {
                        $message = 'Grupo excluído com sucesso!';
                        $message_type = 'success';
                    } else {
                        $message = 'Erro ao excluir grupo.';
                        $message_type = 'danger';
                    }
                }
                break;
        }
    }
}

// Obter lista de grupos
$grupos = getGroups(false); // Incluir inativos também

$page_title = 'Gerenciar Grupos';
include __DIR__ . '/../includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <!-- Sidebar -->
        <div class="col-md-3 col-lg-2 sidebar">
            <div class="text-center mb-4">
                <h4 class="text-white">
                    <i class="bi bi-whatsapp"></i> WhatsApp Rotator
                </h4>
            </div>
            
            <nav class="nav flex-column">
                <a class="nav-link" href="index.php">
                    <i class="bi bi-speedometer2"></i> Dashboard
                </a>
                <a class="nav-link active" href="grupos.php">
                    <i class="bi bi-collection"></i> Grupos
                </a>
                <a class="nav-link" href="numeros.php">
                    <i class="bi bi-telephone"></i> Números
                </a>
                <a class="nav-link" href="relatorios.php">
                    <i class="bi bi-bar-chart"></i> Relatórios
                </a>
                <hr class="text-white">
                <a class="nav-link" href="?logout=1">
                    <i class="bi bi-box-arrow-right"></i> Sair
                </a>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="col-md-9 col-lg-10 main-content">
            <?php if (!empty($message)): ?>
                <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1>Gerenciar Grupos</h1>
                <button class="btn btn-whatsapp" data-bs-toggle="modal" data-bs-target="#newGroupModal">
                    <i class="bi bi-plus-circle"></i> Novo Grupo
                </button>
            </div>
            
            <!-- Lista de Grupos -->
            <div class="card">
                <div class="card-body">
                    <?php if (empty($grupos)): ?>
                        <div class="text-center py-5">
                            <i class="bi bi-collection" style="font-size: 3rem; color: #dee2e6;"></i>
                            <h4 class="mt-3 text-muted">Nenhum grupo encontrado</h4>
                            <p class="text-muted">Crie seu primeiro grupo para começar a organizar seus números de WhatsApp.</p>
                            <button class="btn btn-whatsapp" data-bs-toggle="modal" data-bs-target="#newGroupModal">
                                <i class="bi bi-plus-circle"></i> Criar Primeiro Grupo
                            </button>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-light">
                                    <tr>
                                        <th>Nome</th>
                                        <th>Descrição</th>
                                        <th>Números</th>
                                        <th>Status</th>
                                        <th>Link</th>
                                        <th>Ações</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($grupos as $grupo): ?>
                                        <?php
                                        // Contar números do grupo
                                        $pdo = getConnection();
                                        $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM numeros WHERE grupo_id = ? AND ativo = 1");
                                        $stmt->execute([$grupo['id']]);
                                        $total_numeros = $stmt->fetch()['total'];
                                        
                                        // Gerar link do grupo
                                        $grupo_hash = generateGroupHash($grupo['id']);
                                        $grupo_link = SITE_URL . '/api/redirect.php?g=' . $grupo_hash;
                                        ?>
                                        <tr>
                                            <td>
                                                <strong><?php echo htmlspecialchars($grupo['nome']); ?></strong>
                                            </td>
                                            <td>
                                                <?php echo htmlspecialchars($grupo['descricao']); ?>
                                            </td>
                                            <td>
                                                <span class="badge bg-info"><?php echo $total_numeros; ?> números</span>
                                            </td>
                                            <td>
                                                <?php if ($grupo['ativo']): ?>
                                                    <span class="badge bg-success">Ativo</span>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary">Inativo</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div class="input-group input-group-sm">
                                                    <input type="text" class="form-control" value="<?php echo $grupo_link; ?>" readonly>
                                                    <button class="btn btn-outline-secondary" type="button" onclick="copyToClipboard('<?php echo $grupo_link; ?>')">
                                                        <i class="bi bi-clipboard"></i>
                                                    </button>
                                                </div>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="toggle_status">
                                                        <input type="hidden" name="grupo_id" value="<?php echo $grupo['id']; ?>">
                                                        <button type="submit" class="btn btn-outline-warning" title="Alternar Status">
                                                            <i class="bi bi-toggle-<?php echo $grupo['ativo'] ? 'on' : 'off'; ?>"></i>
                                                        </button>
                                                    </form>
                                                    
                                                    <a href="numeros.php?grupo_id=<?php echo $grupo['id']; ?>" class="btn btn-outline-primary" title="Ver Números">
                                                        <i class="bi bi-telephone"></i>
                                                    </a>
                                                    
                                                    <form method="POST" style="display: inline;" onsubmit="return confirmDelete('Tem certeza que deseja excluir o grupo \'<?php echo htmlspecialchars($grupo['nome']); ?>\'?')">
                                                        <input type="hidden" name="action" value="delete">
                                                        <input type="hidden" name="grupo_id" value="<?php echo $grupo['id']; ?>">
                                                        <button type="submit" class="btn btn-outline-danger" title="Excluir">
                                                            <i class="bi bi-trash"></i>
                                                        </button>
                                                    </form>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal Novo Grupo -->
<div class="modal fade" id="newGroupModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Criar Novo Grupo</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="create">
                    
                    <div class="mb-3">
                        <label for="nome" class="form-label">Nome do Grupo *</label>
                        <input type="text" class="form-control" id="nome" name="nome" required maxlength="100">
                        <div class="form-text">Nome que identificará este grupo (ex: "Vendas", "Suporte", etc.)</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="descricao" class="form-label">Descrição</label>
                        <textarea class="form-control" id="descricao" name="descricao" rows="3" maxlength="500"></textarea>
                        <div class="form-text">Descrição opcional para identificar a finalidade do grupo</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-whatsapp">
                        <i class="bi bi-plus-circle"></i> Criar Grupo
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include __DIR__ . '/../includes/footer.php'; ?>

