<?php
require_once __DIR__ . 
'/../includes/functions.php';

// Verificar autenticação
if (!isAdminAuthenticated()) {
    header('Location: login.php');
    exit;
}

$message = '';
$message_type = '';
$grupo_selecionado = isset($_GET['grupo_id']) ? (int)$_GET['grupo_id'] : 0;

// Processar ações
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add':
                $numero = $_POST['numero'];
                $nome = sanitize($_POST['nome']);
                $grupo_id = (int)$_POST['grupo_id'];
                $mensagem_personalizada = sanitize($_POST['mensagem_personalizada'] ?? '');
                
                if (!empty($numero) && !empty($nome) && $grupo_id > 0) {
                    if (addNumber($numero, $nome, $grupo_id, $mensagem_personalizada)) {
                        $message = 'Número adicionado com sucesso!';
                        $message_type = 'success';
                        $grupo_selecionado = $grupo_id;
                    } else {
                        $message = 'Erro ao adicionar número. Verifique se o formato está correto.';
                        $message_type = 'danger';
                    }
                } else {
                    $message = 'Todos os campos são obrigatórios.';
                    $message_type = 'warning';
                }
                break;
                
            case 'toggle_status':
                $numero_id = (int)$_POST['numero_id'];
                $pdo = getConnection();
                $stmt = $pdo->prepare("UPDATE numeros SET ativo = NOT ativo WHERE id = ?");
                if ($stmt->execute([$numero_id])) {
                    $message = 'Status do número atualizado!';
                    $message_type = 'success';
                } else {
                    $message = 'Erro ao atualizar status.';
                    $message_type = 'danger';
                }
                break;
                
            case 'delete':
                $numero_id = (int)$_POST['numero_id'];
                $pdo = getConnection();
                $stmt = $pdo->prepare("DELETE FROM numeros WHERE id = ?");
                if ($stmt->execute([$numero_id])) {
                    $message = 'Número excluído com sucesso!';
                    $message_type = 'success';
                } else {
                    $message = 'Erro ao excluir número.';
                    $message_type = 'danger';
                }
                break;
                
            case 'edit':
                $numero_id = (int)$_POST['numero_id'];
                $numero = $_POST['numero'];
                $nome = sanitize($_POST['nome']);
                $mensagem_personalizada = sanitize($_POST['mensagem_personalizada'] ?? '');
                
                $numero_formatado = formatWhatsAppNumber($numero);
                if ($numero_formatado && !empty($nome)) {
                    $pdo = getConnection();
                    $stmt = $pdo->prepare("UPDATE numeros SET numero = ?, nome = ?, mensagem_personalizada = ? WHERE id = ?");
                    if ($stmt->execute([$numero_formatado, $nome, $mensagem_personalizada, $numero_id])) {
                        $message = 'Número atualizado com sucesso!';
                        $message_type = 'success';
                    } else {
                        $message = 'Erro ao atualizar número.';
                        $message_type = 'danger';
                    }
                } else {
                    $message = 'Dados inválidos fornecidos.';
                    $message_type = 'warning';
                }
                break;
        }
    }
}

// Obter lista de grupos
$grupos = getGroups(true);

// Obter números do grupo selecionado
$numeros = [];
if ($grupo_selecionado > 0) {
    $numeros = getNumbersByGroup($grupo_selecionado, false);
}

$page_title = 'Gerenciar Números';
include __DIR__ . '/../includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <!-- Sidebar -->
        <div class="col-md-3 col-lg-2 sidebar">
            <div class="text-center mb-4">
                <h4 class="text-white">
                    <i class="bi bi-whatsapp"></i> WhatsApp Rotator
                </h4>
            </div>
            
            <nav class="nav flex-column">
                <a class="nav-link" href="index.php">
                    <i class="bi bi-speedometer2"></i> Dashboard
                </a>
                <a class="nav-link" href="grupos.php">
                    <i class="bi bi-collection"></i> Grupos
                </a>
                <a class="nav-link active" href="numeros.php">
                    <i class="bi bi-telephone"></i> Números
                </a>
                <a class="nav-link" href="relatorios.php">
                    <i class="bi bi-bar-chart"></i> Relatórios
                </a>
                <hr class="text-white">
                <a class="nav-link" href="?logout=1">
                    <i class="bi bi-box-arrow-right"></i> Sair
                </a>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="col-md-9 col-lg-10 main-content">
            <?php if (!empty($message)): ?>
                <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1>Gerenciar Números</h1>
                <button class="btn btn-whatsapp" data-bs-toggle="modal" data-bs-target="#addNumberModal">
                    <i class="bi bi-telephone-plus"></i> Adicionar Número
                </button>
            </div>
            
            <!-- Filtro por Grupo -->
            <div class="card mb-4">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <label for="grupoFilter" class="form-label">Filtrar por Grupo:</label>
                            <select class="form-select" id="grupoFilter" onchange="filterByGroup()">
                                <option value="0">Todos os grupos</option>
                                <?php foreach ($grupos as $grupo): ?>
                                    <option value="<?php echo $grupo['id']; ?>" <?php echo $grupo_selecionado == $grupo['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($grupo['nome']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-4 text-end">
                            <?php if ($grupo_selecionado > 0): ?>
                                <?php
                                $grupo_atual = getGroup($grupo_selecionado);
                                $grupo_hash = generateGroupHash($grupo_selecionado);
                                $grupo_link = SITE_URL . '/api/redirect.php?g=' . $grupo_hash;
                                ?>
                                <small class="text-muted d-block">Link do Grupo:</small>
                                <div class="input-group input-group-sm">
                                    <input type="text" class="form-control" value="<?php echo $grupo_link; ?>" readonly>
                                    <button class="btn btn-outline-secondary" onclick="copyToClipboard('<?php echo $grupo_link; ?>')">
                                        <i class="bi bi-clipboard"></i>
                                    </button>
                                </div>
                                <small class="text-muted d-block mt-2">Exemplo com origem (tráfego pago):</small>
                                <div class="input-group input-group-sm">
                                    <input type="text" class="form-control" value="<?php echo $grupo_link; ?>&origem=trafego_pago" readonly>
                                    <button class="btn btn-outline-secondary" onclick="copyToClipboard('<?php echo $grupo_link; ?>&origem=trafego_pago')">
                                        <i class="bi bi-clipboard"></i>
                                    </button>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Lista de Números -->
            <div class="card">
                <div class="card-body">
                    <?php if (empty($grupos)): ?>
                        <div class="text-center py-5">
                            <i class="bi bi-collection" style="font-size: 3rem; color: #dee2e6;"></i>
                            <h4 class="mt-3 text-muted">Nenhum grupo encontrado</h4>
                            <p class="text-muted">Você precisa criar pelo menos um grupo antes de adicionar números.</p>
                            <a href="grupos.php" class="btn btn-whatsapp">
                                <i class="bi bi-plus-circle"></i> Criar Primeiro Grupo
                            </a>
                        </div>
                    <?php elseif ($grupo_selecionado == 0): ?>
                        <div class="text-center py-5">
                            <i class="bi bi-telephone" style="font-size: 3rem; color: #dee2e6;"></i>
                            <h4 class="mt-3 text-muted">Selecione um grupo</h4>
                            <p class="text-muted">Escolha um grupo no filtro acima para visualizar e gerenciar seus números.</p>
                        </div>
                    <?php elseif (empty($numeros)): ?>
                        <div class="text-center py-5">
                            <i class="bi bi-telephone" style="font-size: 3rem; color: #dee2e6;"></i>
                            <h4 class="mt-3 text-muted">Nenhum número encontrado</h4>
                            <p class="text-muted">Adicione números de WhatsApp a este grupo para começar a usar a rotação.</p>
                            <button class="btn btn-whatsapp" data-bs-toggle="modal" data-bs-target="#addNumberModal">
                                <i class="bi bi-telephone-plus"></i> Adicionar Primeiro Número
                            </button>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-light">
                                    <tr>
                                        <th>Nome</th>
                                        <th>Número</th>
                                        <th>Mensagem</th>
                                        <th>Status</th>
                                        <th>Cliques</th>
                                        <th>Último Clique</th>
                                        <th>Ações</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($numeros as $numero): ?>
                                        <?php
                                        // Obter estatísticas do número
                                        $pdo = getConnection();
                                        $stmt = $pdo->prepare("SELECT COUNT(*) as total, MAX(data_clique) as ultimo_clique FROM cliques WHERE numero_id = ?");
                                        $stmt->execute([$numero['id']]);
                                        $stats = $stmt->fetch();
                                        ?>
                                        <tr>
                                            <td>
                                                <strong><?php echo htmlspecialchars($numero['nome']); ?></strong>
                                            </td>
                                            <td>
                                                <code><?php echo formatWhatsAppNumber($numero['numero']); ?></code>
                                                <a href="<?php echo generateWhatsAppURL($numero['numero'], $numero['mensagem_personalizada']); ?>" target="_blank" class="btn btn-sm btn-outline-success ms-2">
                                                    <i class="bi bi-whatsapp"></i>
                                                </a>
                                            </td>
                                            <td>
                                                <small class="text-muted"><?php echo !empty($numero['mensagem_personalizada']) ? htmlspecialchars(substr($numero['mensagem_personalizada'], 0, 50)) . '...' : 'Nenhuma'; ?></small>
                                            </td>
                                            <td>
                                                <?php if ($numero['ativo']): ?>
                                                    <span class="badge bg-success">Ativo</span>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary">Inativo</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="badge bg-info"><?php echo $stats['total']; ?></span>
                                            </td>
                                            <td>
                                                <?php if ($stats['ultimo_clique']): ?>
                                                    <small class="text-muted">
                                                        <?php echo date('d/m/Y H:i', strtotime($stats['ultimo_clique'])); ?>
                                                    </small>
                                                <?php else: ?>
                                                    <small class="text-muted">Nunca</small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <button class="btn btn-outline-primary" onclick="editNumber(<?php echo htmlspecialchars(json_encode($numero)); ?>)" title="Editar">
                                                        <i class="bi bi-pencil"></i>
                                                    </button>
                                                    
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="toggle_status">
                                                        <input type="hidden" name="numero_id" value="<?php echo $numero['id']; ?>">
                                                        <button type="submit" class="btn btn-outline-warning" title="Alternar Status">
                                                            <i class="bi bi-toggle-<?php echo $numero['ativo'] ? 'on' : 'off'; ?>"></i>
                                                        </button>
                                                    </form>
                                                    
                                                    <form method="POST" style="display: inline;" onsubmit="return confirmDelete('Tem certeza que deseja excluir o número \'<?php echo htmlspecialchars($numero['nome']); ?>\'?')">
                                                        <input type="hidden" name="action" value="delete">
                                                        <input type="hidden" name="numero_id" value="<?php echo $numero['id']; ?>">
                                                        <button type="submit" class="btn btn-outline-danger" title="Excluir">
                                                            <i class="bi bi-trash"></i>
                                                        </button>
                                                    </form>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal Adicionar Número -->
<div class="modal fade" id="addNumberModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Adicionar Número</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add">
                    
                    <div class="mb-3">
                        <label for="grupo_id" class="form-label">Grupo *</label>
                        <select class="form-select" id="grupo_id" name="grupo_id" required>
                            <option value="">Selecione um grupo</option>
                            <?php foreach ($grupos as $grupo): ?>
                                <option value="<?php echo $grupo['id']; ?>" <?php echo $grupo_selecionado == $grupo['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($grupo['nome']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="nome" class="form-label">Nome/Identificação *</label>
                        <input type="text" class="form-control" id="nome" name="nome" required maxlength="100">
                        <div class="form-text">Nome para identificar este número (ex: "João - Vendas")</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="numero" class="form-label">Número do WhatsApp *</label>
                        <input type="text" class="form-control" id="numero" name="numero" required data-mask="phone" placeholder="(11) 99999-9999">
                        <div class="form-text">Digite apenas números. O código do país (55) será adicionado automaticamente se necessário.</div>
                    </div>

                    <div class="mb-3">
                        <label for="mensagem_personalizada" class="form-label">Mensagem Personalizada (Opcional)</label>
                        <textarea class="form-control" id="mensagem_personalizada" name="mensagem_personalizada" rows="3" maxlength="500"></textarea>
                        <div class="form-text">Esta mensagem será pré-preenchida no WhatsApp quando o cliente clicar no link.</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-whatsapp">
                        <i class="bi bi-telephone-plus"></i> Adicionar Número
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal Editar Número -->
<div class="modal fade" id="editNumberModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Editar Número</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="numero_id" id="edit_numero_id">
                    
                    <div class="mb-3">
                        <label for="edit_nome" class="form-label">Nome/Identificação *</label>
                        <input type="text" class="form-control" id="edit_nome" name="nome" required maxlength="100">
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit_numero" class="form-label">Número do WhatsApp *</label>
                        <input type="text" class="form-control" id="edit_numero" name="numero" required data-mask="phone">
                        <div class="form-text">Digite apenas números. O código do país (55) será adicionado automaticamente se necessário.</div>
                    </div>

                    <div class="mb-3">
                        <label for="edit_mensagem_personalizada" class="form-label">Mensagem Personalizada (Opcional)</label>
                        <textarea class="form-control" id="edit_mensagem_personalizada" name="mensagem_personalizada" rows="3" maxlength="500"></textarea>
                        <div class="form-text">Esta mensagem será pré-preenchida no WhatsApp quando o cliente clicar no link.</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-whatsapp">
                        <i class="bi bi-check-circle"></i> Salvar Alterações
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
    function filterByGroup() {
        const grupoId = document.getElementById('grupoFilter').value;
        window.location.href = 'numeros.php' + (grupoId > 0 ? '?grupo_id=' + grupoId : '');
    }
    
    function editNumber(numero) {
        document.getElementById('edit_numero_id').value = numero.id;
        document.getElementById('edit_nome').value = numero.nome;
        document.getElementById('edit_numero').value = numero.numero;
        document.getElementById('edit_mensagem_personalizada').value = numero.mensagem_personalizada;
        
        const modal = new bootstrap.Modal(document.getElementById('editNumberModal'));
        modal.show();
    }

    function confirmDelete(message) {
        return confirm(message);
    }

    function copyToClipboard(text) {
        navigator.clipboard.writeText(text).then(function() {
            alert('Link copiado para a área de transferência!');
        }, function(err) {
            console.error('Erro ao copiar: ', err);
            alert('Erro ao copiar o link. Por favor, copie manualmente.');
        });
    }
</script>

<?php include __DIR__ . '/../includes/footer.php'; ?>

