<?php
require_once __DIR__ . '/../config/database.php';

// Função para sanitizar entrada
function sanitize($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

// Função para validar número de WhatsApp
function validateWhatsAppNumber($number) {
    // Remove todos os caracteres não numéricos
    $number = preg_replace('/[^0-9]/', '', $number);
    
    // Verifica se tem pelo menos 10 dígitos (formato mínimo)
    if (strlen($number) < 10 || strlen($number) > 15) {
        return false;
    }
    
    return $number;
}

// Função para formatar número de WhatsApp
function formatWhatsAppNumber($number) {
    $number = validateWhatsAppNumber($number);
    if (!$number) return false;
    
    // Se não começar com código do país, adiciona 55 (Brasil)
    if (strlen($number) == 11 && substr($number, 0, 2) != '55') {
        $number = '55' . $number;
    }
    
    return $number;
}

// Função para gerar URL do WhatsApp
function generateWhatsAppURL($number, $message = '') {
    $number = formatWhatsAppNumber($number);
    if (!$number) return false;
    
    $url = "https://wa.me/" . $number;
    if (!empty($message)) {
        $url .= "?text=" . urlencode($message);
    }
    
    return $url;
}

// Função para criar grupo
function createGroup($nome, $descricao = '') {
    $pdo = getConnection();
    $stmt = $pdo->prepare("INSERT INTO grupos (nome, descricao) VALUES (?, ?)");
    return $stmt->execute([sanitize($nome), sanitize($descricao)]);
}

// Função para listar grupos
function getGroups($ativo_apenas = true) {
    $pdo = getConnection();
    $sql = "SELECT * FROM grupos";
    if ($ativo_apenas) {
        $sql .= " WHERE ativo = 1";
    }
    $sql .= " ORDER BY nome";
    
    $stmt = $pdo->query($sql);
    return $stmt->fetchAll();
}

// Função para obter grupo por ID
function getGroup($id) {
    $pdo = getConnection();
    $stmt = $pdo->prepare("SELECT * FROM grupos WHERE id = ?");
    $stmt->execute([$id]);
    return $stmt->fetch();
}

// Função para adicionar número
function addNumber($numero, $nome, $grupo_id, $mensagem_personalizada = '') {
    $numero = formatWhatsAppNumber($numero);
    if (!$numero) return false;
    
    $pdo = getConnection();
    $stmt = $pdo->prepare("INSERT INTO numeros (numero, nome, grupo_id, mensagem_personalizada) VALUES (?, ?, ?, ?)");
    return $stmt->execute([$numero, sanitize($nome), $grupo_id, sanitize($mensagem_personalizada)]);
}

// Função para listar números de um grupo
function getNumbersByGroup($grupo_id, $ativo_apenas = true) {
    $pdo = getConnection();
    $sql = "SELECT * FROM numeros WHERE grupo_id = ?";
    if ($ativo_apenas) {
        $sql .= " AND ativo = 1";
    }
    $sql .= " ORDER BY nome";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$grupo_id]);
    return $stmt->fetchAll();
}

// Função para obter próximo número na rotação (Round-Robin estrito)
function getNextNumber($grupo_id) {
    $pdo = getConnection();
    
    // Buscar números ativos do grupo ordenados por ID para consistência
    $stmt = $pdo->prepare("SELECT * FROM numeros WHERE grupo_id = ? AND ativo = 1 ORDER BY id ASC");
    $stmt->execute([$grupo_id]);
    $numeros = $stmt->fetchAll();
    
    if (empty($numeros)) {
        return false;
    }
    
    // Se há apenas um número, sempre retorna ele
    if (count($numeros) == 1) {
        return $numeros[0];
    }
    
    // Buscar estado atual da rotação
    $stmt = $pdo->prepare("SELECT * FROM rotacao_estado WHERE grupo_id = ?");
    $stmt->execute([$grupo_id]);
    $estado = $stmt->fetch();
    
    if (!$estado) {
        // Primeiro acesso, criar estado inicial com o primeiro número
        $primeiro_numero = $numeros[0];
        $stmt = $pdo->prepare("INSERT INTO rotacao_estado (grupo_id, ultimo_numero_id, contador) VALUES (?, ?, 1)");
        $stmt->execute([$grupo_id, $primeiro_numero['id']]);
        return $primeiro_numero;
    }
    
    // Encontrar posição atual do último número usado
    $posicao_atual = -1;
    foreach ($numeros as $index => $numero) {
        if ($numero['id'] == $estado['ultimo_numero_id']) {
            $posicao_atual = $index;
            break;
        }
    }
    
    // Se o último número usado não existe mais (foi desativado), começar do primeiro
    if ($posicao_atual == -1) {
        $posicao_atual = -1; // Será incrementado para 0
    }
    
    // Próxima posição (circular) - sempre avança para o próximo
    $proxima_posicao = ($posicao_atual + 1) % count($numeros);
    $proximo_numero = $numeros[$proxima_posicao];
    
    // Atualizar estado da rotação
    $stmt = $pdo->prepare("UPDATE rotacao_estado SET ultimo_numero_id = ?, contador = contador + 1 WHERE grupo_id = ?");
    $stmt->execute([$proximo_numero['id'], $grupo_id]);
    
    return $proximo_numero;
}

// Função para registrar clique
function registerClick($grupo_id, $numero_id, $ip_address, $user_agent, $origem = '') {
    $pdo = getConnection();
    $stmt = $pdo->prepare("INSERT INTO cliques (grupo_id, numero_id, ip_address, user_agent, origem) VALUES (?, ?, ?, ?, ?)");
    return $stmt->execute([$grupo_id, $numero_id, $ip_address, $user_agent, sanitize($origem)]);
}

// Função para gerar hash único do grupo (para URL)
function generateGroupHash($grupo_id) {
    return substr(md5($grupo_id . SECRET_KEY), 0, 8);
}

// Função para obter grupo por hash
function getGroupByHash($hash) {
    $grupos = getGroups();
    foreach ($grupos as $grupo) {
        if (generateGroupHash($grupo['id']) === $hash) {
            return $grupo;
        }
    }
    return false;
}

// Função para obter estatísticas de um grupo
function getGroupStats($grupo_id, $dias = 30) {
    $pdo = getConnection();
    
    // Total de cliques
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM cliques WHERE grupo_id = ? AND data_clique >= DATE_SUB(NOW(), INTERVAL {$dias} DAY) ");
    $stmt->execute([$grupo_id]);
    $total_cliques = $stmt->fetch()['total'];
    
    // Cliques por número
    $stmt = $pdo->prepare("
        SELECT n.nome, n.numero, COUNT(c.id) as cliques 
        FROM numeros n 
        LEFT JOIN cliques c ON n.id = c.numero_id AND c.data_clique >= DATE_SUB(NOW(), INTERVAL {$dias} DAY) 
        WHERE n.grupo_id = ? 
        GROUP BY n.id 
        ORDER BY cliques DESC
    ");
    $stmt->execute([$grupo_id]);
    $cliques_por_numero = $stmt->fetchAll();
    
    return [
        'total_cliques' => $total_cliques,
        'cliques_por_numero' => $cliques_por_numero
    ];
}

// Função para obter IP do cliente
function getClientIP() {
    $ip_keys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
    foreach ($ip_keys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

// Função para verificar autenticação admin (simples)
function isAdminAuthenticated() {
    session_start();
    return isset($_SESSION['admin_authenticated']) && $_SESSION['admin_authenticated'] === true;
}

// Função para autenticar admin
function authenticateAdmin($username, $password) {
    $pdo = getConnection();
    $stmt = $pdo->prepare("SELECT * FROM admins WHERE username = ?");
    $stmt->execute([$username]);
    $admin = $stmt->fetch();

    if ($admin && password_verify($password, $admin["password"])) {
        session_start();
        $_SESSION["admin_authenticated"] = true;
        $_SESSION["admin_username"] = $admin["username"];
        $_SESSION["admin_id"] = $admin["id"];
        return true;
    }
    return false;
}

// Função para logout admin
function logoutAdmin() {
    session_start();
    session_destroy();
}

// Função para obter estatísticas de cliques por origem
function getClicksByOrigin($grupo_id, $dias = 30) {
    $pdo = getConnection();
    $stmt = $pdo->prepare("
        SELECT origem, COUNT(*) as total_cliques
        FROM cliques
        WHERE grupo_id = ? AND data_clique >= DATE_SUB(NOW(), INTERVAL {$dias} DAY) 
        GROUP BY origem
        ORDER BY total_cliques DESC
    ");
    $stmt->execute([$grupo_id]);
    return $stmt->fetchAll();
}

